package gov.cms.grouper.snf.component.v100.logic.nursing;

import static gov.cms.grouper.snf.component.v100.TestUtil.of;
import static gov.cms.grouper.snf.model.enums.NursingCmg.BAB1;
import static gov.cms.grouper.snf.model.enums.NursingCmg.BAB2;
import static gov.cms.grouper.snf.model.enums.NursingCmg.CBC2;
import static org.mockito.ArgumentMatchers.anyBoolean;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.spy;

import gov.cms.grouper.snf.component.v100.TestUtil;
import gov.cms.grouper.snf.lego.SnfCache;
import gov.cms.grouper.snf.lego.SnfCache.Reference;
import gov.cms.grouper.snf.lego.SnfUtils;
import gov.cms.grouper.snf.model.Assessment;
import gov.cms.grouper.snf.model.enums.NursingCmg;
import gov.cms.grouper.snf.model.reader.Rai300;
import gov.cms.grouper.snf.util.ClaimInfo;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

public class BscpLogicTest {

  public static final int functionScore = 12;


  public static BscpLogic make(int version, ClaimInfo claim, Supplier<NursingCmg> ph) {
    Reference<ReducedPhysicalFunctionLogic> physicRef = SnfCache.of();

    final BscpLogic logic = new BscpLogic(claim, physicRef);
    BscpLogic spy = spy(logic);

    ReducedPhysicalFunctionLogic physical = new ReducedPhysicalFunctionLogic(claim, spy);
    physical = spy(physical);
    physicRef.set(physical);
    doReturn((ph == null) ? null : ph.get()).when(physical).exec();


    return spy;
  }



  @Test
  public void testEvaluateBehavioralSymptomsCognitivePerformance() {
    int version = 100;

    ClaimInfo claim = of(10);
    BscpLogic logic = make(version, claim, () -> null);
    Assertions.assertEquals(BAB1, logic.eval(false, () -> 0, () -> 1, () -> BAB1));

    claim = of(12);
    logic = make(version, claim, () -> null);
    doReturn(BAB2).when(logic).step4(anyBoolean());
    Assertions.assertEquals(BAB2, logic.eval(false, () -> 0, () -> 0, () -> null));

    claim = of(12);
    logic = make(version, claim, null);
    doReturn(BAB1).when(logic).step6();
    Assertions.assertEquals(BAB1, logic.eval(false, () -> 9, () -> 1, () -> null));

    claim = of(12);
    logic = make(version, claim, null);
    doReturn(BAB1).when(logic).step4(anyBoolean());
    Assertions.assertEquals(BAB1, logic.eval(false, () -> 10, () -> 1, () -> null));

    claim = of(12);
    logic = make(version, claim, null);
    doReturn(1).when(logic).step5();
    Assertions.assertEquals(BAB1, logic.eval(true, () -> 99, () -> 1, () -> null));

    claim = of(12);
    logic = make(version, claim, null);
    doReturn(0).when(logic).step5();
    doReturn(BAB1).when(logic).step4(anyBoolean());
    Assertions.assertEquals(BAB1, logic.eval(false, () -> 99, () -> 1, () -> null));

    claim = of(12);
    logic = make(version, claim, null);
    doReturn(3).when(logic).step5();
    Assertions.assertEquals(BAB2, logic.eval(false, () -> 8, () -> 1, () -> null));

  }

  @Test
  public void testBehavioralSymptomsCognitivePerformanceStep4() {

    List<Assessment> assessments =
        Collections.singletonList(new Assessment("", Rai300.E0100A.name(), 1));
    ClaimInfo claim = of(assessments, 12);

    BscpLogic logic = make(100, claim, () -> BAB2);

    List<Assessment> behavioralSymptomsAssessmentList =
        logic.getBehavioralSymptomsAssessmentList(claim);
    boolean e0100AChecked = claim.isCheckedAndNotNull(Rai300.E0100A);
    boolean e0100BChecked = claim.isCheckedAndNotNull(Rai300.E0100B);
    boolean hasSymptoms = logic.hasSymptoms(behavioralSymptomsAssessmentList);

    boolean isBehavioralSymptomsPresent =
        logic.isBehavioralSymptomsPresent(e0100AChecked, e0100BChecked, hasSymptoms);
    doReturn(5).when(logic).step5();

    NursingCmg actual = logic.step4(isBehavioralSymptomsPresent);
    Assertions.assertEquals(BAB2, actual);
  }


  @Test
  public void testBehavioralSymptomsCognitivePerformanceStep5() {
    Set<Assessment> assessments = SnfUtils.toSet(Arrays.asList(
        new Assessment(Rai300.O0500A.toString(), Rai300.O0500A.toString(),
            Const.nursingCountNumDaysInLast7Days),
        new Assessment(Rai300.O0500B.toString(), Rai300.O0500B.toString(),
            Const.nursingCountNumDaysInLast7Days),
        new Assessment(Rai300.O0500D.toString(), Rai300.O0500D.toString(),
            Const.nursingCountNumDaysInLast7Days),
        new Assessment(Rai300.O0500F.toString(), Rai300.O0500F.toString(),
            Const.nursingCountNumDaysInLast7Days),
        new Assessment(Rai300.H0200C.toString(), Rai300.H0200C.toString(), 1),
        new Assessment(Rai300.H0500.toString(), Rai300.H0500.toString(), 1)));
    Map<Rai300, Assessment> testMap = new HashMap<>();
    for (Assessment ast : assessments) {
      testMap.put(Rai300.valueOf(ast.getName()), ast);
    }

    ClaimInfo claim = of(new ArrayList<>(testMap.values()));
    BscpLogic logic = make(100, claim, null);
    int expected = 3;
    int actual = logic.step5();
    Assertions.assertEquals(expected, actual);

    // testMap.put(Rai300.H0200C, Const.nursingCountNumDaysInLast7Days - 1);
    testMap.put(Rai300.H0200C,
        new Assessment(Rai300.H0200C.toString(), Rai300.H0200C.toString(), 0));
    claim = of(new ArrayList<>(testMap.values()));
    logic = make(100, claim, null);
    expected = 3;
    actual = logic.step5();
    Assertions.assertEquals(expected, actual);

    // testMap.put(Rai300.H0500, Const.nursingCountNumDaysInLast7Days - 1);
    testMap.put(Rai300.H0500, new Assessment(Rai300.H0500.toString(), Rai300.H0500.toString(), 0));
    claim = of(new ArrayList<>(testMap.values()));
    logic = make(100, claim, null);
    expected = 2;
    actual = logic.step5();
    Assertions.assertEquals(expected, actual);

    // testMap.put(Rai300.O0500A, Const.nursingCountNumDaysInLast7Days - 1);
    testMap.put(Rai300.O0500A, new Assessment(Rai300.O0500A.toString(), Rai300.O0500A.toString(),
        Const.nursingCountNumDaysInLast7Days - 1));
    claim = of(new ArrayList<>(testMap.values()));
    logic = make(100, claim, null);
    expected = 2;
    actual = logic.step5();
    Assertions.assertEquals(expected, actual);

    // testMap.put(Rai300.O0500B, Const.nursingCountNumDaysInLast7Days - 1);
    testMap.put(Rai300.O0500B, new Assessment(Rai300.O0500B.toString(), Rai300.O0500B.toString(),
        Const.nursingCountNumDaysInLast7Days - 1));
    claim = of(new ArrayList<>(testMap.values()));
    logic = make(100, claim, null);
    expected = 1;
    actual = logic.step5();
    Assertions.assertEquals(expected, actual);

    // testMap.put(Rai300.O0500D, Const.nursingCountNumDaysInLast7Days - 1);
    testMap.put(Rai300.O0500D, new Assessment(Rai300.O0500D.toString(), Rai300.O0500D.toString(),
        Const.nursingCountNumDaysInLast7Days - 1));
    claim = of(new ArrayList<>(testMap.values()));
    logic = make(100, claim, null);
    expected = 1;
    actual = logic.step5();
    Assertions.assertEquals(expected, actual);

    // testMap.put(Rai300.O0500F, Const.nursingCountNumDaysInLast7Days - 1);
    testMap.put(Rai300.O0500F, new Assessment(Rai300.O0500F.toString(), Rai300.O0500F.toString(),
        Const.nursingCountNumDaysInLast7Days - 1));
    claim = of(new ArrayList<>(testMap.values()));
    logic = make(100, claim, null);
    expected = 0;
    actual = logic.step5();
    Assertions.assertEquals(expected, actual);

    // testMap.put(BscpLogic.Services.get(0), Const.nursingCountNumDaysInLast7Days);
    testMap.put(BscpLogic.Services.get(0), new Assessment(BscpLogic.Services.get(0).toString(),
        BscpLogic.Services.get(0).toString(), Const.nursingCountNumDaysInLast7Days));
    claim = of(new ArrayList<>(testMap.values()));
    logic = make(100, claim, null);
    expected = 1;
    actual = logic.step5();
    Assertions.assertEquals(expected, actual);

    // testMap.put(BscpLogic.Services.get(0), Const.nursingCountNumDaysInLast7Days + 1);
    testMap.put(BscpLogic.Services.get(0), new Assessment(BscpLogic.Services.get(0).toString(),
        BscpLogic.Services.get(0).toString(), Const.nursingCountNumDaysInLast7Days + 1));
    claim = of(new ArrayList<>(testMap.values()));
    logic = make(100, claim, null);
    expected = 1;
    actual = logic.step5();
    Assertions.assertEquals(expected, actual);

    // testMap.put(BscpLogic.Services.get(0), Const.nursingCountNumDaysInLast7Days - 1);
    testMap.put(BscpLogic.Services.get(0), new Assessment(BscpLogic.Services.get(0).toString(),
        BscpLogic.Services.get(0).toString(), Const.nursingCountNumDaysInLast7Days - 1));
    claim = of(new ArrayList<>(testMap.values()));
    logic = make(100, claim, null);
    expected = 0;
    actual = logic.step5();
    Assertions.assertEquals(expected, actual);

    // testMap.put(Rai300.O0500A, Const.nursingCountNumDaysInLast7Days + 1);
    testMap.put(Rai300.O0500A, new Assessment(Rai300.O0500A.toString(), Rai300.O0500A.toString(),
        Const.nursingCountNumDaysInLast7Days + 1));
    // testMap.put(Rai300.O0500B, Const.nursingCountNumDaysInLast7Days + 1);
    testMap.put(Rai300.O0500B, new Assessment(Rai300.O0500B.toString(), Rai300.O0500B.toString(),
        Const.nursingCountNumDaysInLast7Days + 1));
    // testMap.put(Rai300.O0500D, Const.nursingCountNumDaysInLast7Days + 1);
    testMap.put(Rai300.O0500D, new Assessment(Rai300.O0500D.toString(), Rai300.O0500D.toString(),
        Const.nursingCountNumDaysInLast7Days + 1));
    // testMap.put(Rai300.O0500F, Const.nursingCountNumDaysInLast7Days + 1);
    testMap.put(Rai300.O0500F, new Assessment(Rai300.O0500F.toString(), Rai300.O0500F.toString(),
        Const.nursingCountNumDaysInLast7Days + 1));
    // testMap.put(Rai300.H0200C, Const.nursingCountNumDaysInLast7Days + 1);
    testMap.put(Rai300.H0200C,
        new Assessment(Rai300.H0200C.toString(), Rai300.H0200C.toString(), 1));
    // testMap.put(Rai300.H0500, Const.nursingCountNumDaysInLast7Days + 1);
    testMap.put(Rai300.H0500, new Assessment(Rai300.H0500.toString(), Rai300.H0500.toString(), 1));

    claim = of(new ArrayList<>(testMap.values()));
    logic = make(100, claim, null);
    expected = 3;
    actual = logic.step5();
    Assertions.assertEquals(expected, actual);


  }

  @Test
  public void testBehavioralSymptomsCognitivePerformanceStep6() {
    ClaimInfo claim = of(4);
    BscpLogic logic = make(100, claim, null);
    doReturn(3).when(logic).step5();
    Assertions.assertNull(logic.step6());

    claim = of(12);
    logic = make(100, claim, null);
    doReturn(3).when(logic).step5();
    Assertions.assertEquals(BAB2, logic.step6());

    claim = of(12);
    logic = make(100, claim, null);
    doReturn(1).when(logic).step5();
    Assertions.assertEquals(BAB1, logic.step6());
  }

  @Test
  public void testExec() {
    List<Assessment> assessments =
        Arrays.asList(new Assessment(Rai300.O0500B.name(), Rai300.O0500B.name(), 8),
            new Assessment(Rai300.O0500J.name(), Rai300.O0500J.name(), 7));
    ClaimInfo claim = of(assessments);
    BscpLogic logic = make(100, claim, () -> CBC2);
    NursingCmg actual = logic.exec();
    NursingCmg expected = CBC2;
    Assertions.assertEquals(expected, actual);

    claim = of(TestUtil.getAll(1));
    logic = make(100, claim, () -> CBC2);
    actual = logic.exec();
    expected = CBC2;
    Assertions.assertEquals(expected, actual);

  }

  public static void main(String[] args) {
    List<Assessment> assessments =
        Arrays.asList(new Assessment(Rai300.O0500B.name(), Rai300.O0500B.name(), 8),
            new Assessment(Rai300.O0500J.name(), Rai300.O0500J.name(), 7));
    ClaimInfo claim = of(assessments);
    make(100, claim, SnfCache.ofReference(NursingCmg.BAB1));
  }

}
